<?php
/**
 * Auth.php
 * Gestión de autenticación y sesiones
 */

class Auth {
    private $pdo;

    public function __construct($pdo) {
        $this->pdo = $pdo;
    }

    /**
     * Registrar un nuevo usuario
     */
    public function register($nombre, $email, $password, $rol = 'editor') {
        // Validar entrada
        if (empty($nombre) || empty($email) || empty($password)) {
            return ['success' => false, 'message' => 'Todos los campos son requeridos'];
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return ['success' => false, 'message' => 'Email inválido'];
        }

        if (strlen($password) < 8) {
            return ['success' => false, 'message' => 'La contraseña debe tener al menos 8 caracteres'];
        }

        try {
            // Verificar si el email ya existe
            $stmt = $this->pdo->prepare('SELECT id FROM usuarios WHERE email = ?');
            $stmt->execute([$email]);
            
            if ($stmt->rowCount() > 0) {
                return ['success' => false, 'message' => 'El email ya está registrado'];
            }

            // Insertar nuevo usuario
            $hashed_password = password_hash($password, PASSWORD_BCRYPT, ['cost' => 10]);
            $stmt = $this->pdo->prepare('
                INSERT INTO usuarios (nombre, email, password, rol) 
                VALUES (?, ?, ?, ?)
            ');
            $stmt->execute([$nombre, $email, $hashed_password, $rol]);

            return ['success' => true, 'message' => 'Usuario registrado exitosamente'];
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Error en el registro: ' . $e->getMessage()];
        }
    }

    /**
     * Iniciar sesión
     */
    public function login($email, $password) {
        // Validar entrada
        if (empty($email) || empty($password)) {
            return ['success' => false, 'message' => 'Email y contraseña requeridos'];
        }

        try {
            $stmt = $this->pdo->prepare('SELECT id, nombre, email, password, rol FROM usuarios WHERE email = ? AND activo = 1');
            $stmt->execute([$email]);
            $user = $stmt->fetch();

            if (!$user || !password_verify($password, $user['password'])) {
                $this->recordFailedLogin($email);
                return ['success' => false, 'message' => 'Email o contraseña incorrectos'];
            }

            // Crear sesión
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['usuario'] = $user['email'];
            $_SESSION['rol'] = $user['rol'];
            $_SESSION['login_time'] = time();

            $this->logAcceso($user['id'], 'login', 'success');

            return ['success' => true, 'message' => 'Sesión iniciada correctamente'];
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Error en el login: ' . $e->getMessage()];
        }
    }

    /**
     * Cerrar sesión
     */
    public function logout() {
        if (isset($_SESSION['user_id'])) {
            $this->logAcceso($_SESSION['user_id'], 'logout', 'success');
        }
        session_destroy();
        return ['success' => true, 'message' => 'Sesión cerrada'];
    }

    /**
     * Verificar si el usuario está autenticado
     */
    public function isAuthenticated() {
        return isset($_SESSION['user_id']) && isset($_SESSION['usuario']);
    }

    /**
     * Verificar sesión expirada
     */
    public function checkSessionTimeout() {
        if ($this->isAuthenticated()) {
            $elapsed = time() - $_SESSION['login_time'];
            if ($elapsed > SESSION_TIMEOUT) {
                $this->logout();
                return false;
            }
            $_SESSION['login_time'] = time();
        }
        return true;
    }

    /**
     * Obtener ID del usuario actual
     */
    public function getCurrentUserId() {
        return $_SESSION['user_id'] ?? null;
    }

    /**
     * Obtener email del usuario actual
     */
    public function getCurrentUserEmail() {
        return $_SESSION['usuario'] ?? null;
    }

    /**
     * Obtener rol del usuario actual
     */
    public function getCurrentUserRole() {
        return $_SESSION['rol'] ?? null;
    }

    /**
     * Verificar si es administrador
     */
    public function isAdmin() {
        return $this->getCurrentUserRole() === 'admin';
    }

    /**
     * Registrar acceso
     */
    private function logAcceso($user_id, $accion, $estado) {
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
        
        $stmt = $this->pdo->prepare('
            INSERT INTO logs_acceso (usuario_id, accion, ip_origen, user_agent, estado) 
            VALUES (?, ?, ?, ?, ?)
        ');
        $stmt->execute([$user_id, $accion, $ip, $user_agent, $estado]);
    }

    /**
     * Registrar intento fallido de login
     */
    private function recordFailedLogin($email) {
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        
        $stmt = $this->pdo->prepare('
            SELECT COUNT(*) FROM logs_acceso 
            WHERE accion = ? AND ip_origen = ? 
            AND fecha > DATE_SUB(NOW(), INTERVAL 15 MINUTE)
        ');
        $stmt->execute(['failed_login', $ip]);
        $attempts = $stmt->fetchColumn();

        if ($attempts >= MAX_LOGIN_ATTEMPTS) {
            // Bloquear la IP
            $_SESSION['login_blocked_until'] = time() + LOCKOUT_TIME;
        }
    }

    /**
     * Verificar si está bloqueado por intentos fallidos
     */
    public function isLoginBlocked() {
        if (isset($_SESSION['login_blocked_until'])) {
            if (time() < $_SESSION['login_blocked_until']) {
                return true;
            } else {
                unset($_SESSION['login_blocked_until']);
            }
        }
        return false;
    }
}
?>
