<?php
/**
 * Config.php
 * Gestión de configuración de la aplicación
 */

class ConfigManager {
    private $pdo;
    private $auth;

    public function __construct($pdo, $auth) {
        $this->pdo = $pdo;
        $this->auth = $auth;
    }

    /**
     * Obtener configuración actual
     */
    public function getConfig() {
        try {
            $stmt = $this->pdo->query('SELECT * FROM configuracion WHERE activo = 1 LIMIT 1');
            return $stmt->fetch();
        } catch (Exception $e) {
            return null;
        }
    }

    /**
     * Obtener configuración por ID
     */
    public function getConfigById($id) {
        try {
            $stmt = $this->pdo->prepare('SELECT * FROM configuracion WHERE id = ?');
            $stmt->execute([$id]);
            return $stmt->fetch();
        } catch (Exception $e) {
            return null;
        }
    }

    /**
     * Actualizar configuración
     */
    public function updateConfig($data) {
        if (!$this->auth->isAuthenticated()) {
            return ['success' => false, 'message' => 'No autenticado'];
        }

        try {
            $config_actual = $this->getConfig();
            
            // Preparar datos
            $nombre_estacion = $data['nombre_estacion'] ?? $config_actual['nombre_estacion'];
            $descripcion = $data['descripcion'] ?? $config_actual['descripcion'];
            $url_stream = $data['url_stream'] ?? $config_actual['url_stream'];
            $color_primario = $this->validarColor($data['color_primario'] ?? $config_actual['color_primario']);
            $color_secundario = $this->validarColor($data['color_secundario'] ?? $config_actual['color_secundario']);
            $url_logo = $data['url_logo'] ?? $config_actual['url_logo'];
            $url_facebook = $data['url_facebook'] ?? $config_actual['url_facebook'];
            $url_instagram = $data['url_instagram'] ?? $config_actual['url_instagram'];
            $url_tiktok = $data['url_tiktok'] ?? $config_actual['url_tiktok'];
            $url_whatsapp = $data['url_whatsapp'] ?? $config_actual['url_whatsapp'];

            // Actualizar
            $stmt = $this->pdo->prepare('
                UPDATE configuracion SET
                    nombre_estacion = ?,
                    descripcion = ?,
                    url_stream = ?,
                    color_primario = ?,
                    color_secundario = ?,
                    url_logo = ?,
                    url_facebook = ?,
                    url_instagram = ?,
                    url_tiktok = ?,
                    url_whatsapp = ?
                WHERE id = ?
            ');

            $success = $stmt->execute([
                $nombre_estacion,
                $descripcion,
                $url_stream,
                $color_primario,
                $color_secundario,
                $url_logo,
                $url_facebook,
                $url_instagram,
                $url_tiktok,
                $url_whatsapp,
                $config_actual['id']
            ]);

            if ($success) {
                // Registrar cambio
                $this->registrarCambio($config_actual, [
                    'nombre_estacion' => $nombre_estacion,
                    'descripcion' => $descripcion,
                    'url_stream' => $url_stream,
                    'color_primario' => $color_primario,
                    'color_secundario' => $color_secundario,
                    'url_logo' => $url_logo,
                    'url_facebook' => $url_facebook,
                    'url_instagram' => $url_instagram,
                    'url_tiktok' => $url_tiktok,
                    'url_whatsapp' => $url_whatsapp
                ]);

                return ['success' => true, 'message' => 'Configuración actualizada'];
            } else {
                return ['success' => false, 'message' => 'Error al actualizar'];
            }
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Error: ' . $e->getMessage()];
        }
    }

    /**
     * Validar formato de color hex
     */
    private function validarColor($color) {
        if (preg_match('/^#[a-f0-9]{6}$|^#[a-f0-9]{8}$/i', $color)) {
            return $color;
        }
        if (preg_match('/^#FF[a-f0-9]{6}$/i', $color)) {
            return $color;
        }
        return '#FF6200EE'; // Color por defecto
    }

    /**
     * Registrar cambio en historial
     */
    private function registrarCambio($config_anterior, $config_nueva) {
        $usuario_id = $this->auth->getCurrentUserId();
        $cambios = [];

        foreach ($config_nueva as $key => $value) {
            if ($config_anterior[$key] !== $value) {
                $cambios[$key] = ['anterior' => $config_anterior[$key], 'nuevo' => $value];
            }
        }

        if (count($cambios) > 0) {
            $stmt = $this->pdo->prepare('
                INSERT INTO historial_cambios (usuario_id, tipo_cambio, descripcion, valor_anterior, valor_nuevo)
                VALUES (?, ?, ?, ?, ?)
            ');

            $stmt->execute([
                $usuario_id,
                'config_update',
                'Actualización de configuración',
                json_encode(array_keys($cambios)),
                json_encode($cambios)
            ]);
        }
    }

    /**
     * Obtener historial de cambios
     */
    public function getHistorial($limite = 50) {
        try {
            $stmt = $this->pdo->prepare('
                SELECT h.*, u.nombre as usuario_nombre
                FROM historial_cambios h
                JOIN usuarios u ON h.usuario_id = u.id
                ORDER BY h.fecha_cambio DESC
                LIMIT ?
            ');
            $stmt->execute([$limite]);
            return $stmt->fetchAll();
        } catch (Exception $e) {
            return [];
        }
    }

    /**
     * Subir logo/archivo
     */
    public function uploadFile($file, $tipo = 'logo') {
        if ($file['size'] > MAX_FILE_SIZE) {
            return ['success' => false, 'message' => 'Archivo muy grande'];
        }

        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if (!in_array($ext, ALLOWED_EXTENSIONS)) {
            return ['success' => false, 'message' => 'Tipo de archivo no permitido'];
        }

        $filename = $tipo . '_' . time() . '.' . $ext;
        $filepath = UPLOAD_DIR . $filename;

        if (!is_dir(UPLOAD_DIR)) {
            mkdir(UPLOAD_DIR, 0755, true);
        }

        if (move_uploaded_file($file['tmp_name'], $filepath)) {
            return [
                'success' => true,
                'message' => 'Archivo subido',
                'url' => 'uploads/' . $filename
            ];
        }

        return ['success' => false, 'message' => 'Error al subir archivo'];
    }

    /**
     * Exportar configuración como JSON
     */
    public function exportarJSON() {
        $config = $this->getConfig();
        if (!$config) {
            return null;
        }

        return [
            'stationName' => $config['nombre_estacion'],
            'streamUrl' => $config['url_stream'],
            'primaryColor' => $config['color_primario'],
            'secondaryColor' => $config['color_secundario'],
            'logoUrl' => $config['url_logo'],
            'facebookUrl' => $config['url_facebook'],
            'instagramUrl' => $config['url_instagram'],
            'tiktokUrl' => $config['url_tiktok'],
            'whatsappUrl' => $config['url_whatsapp'],
            'description' => $config['descripcion']
        ];
    }
}
?>
