<?php
/**
 * index.php
 * Panel de administración principal
 */

require_once('config.php');
require_once('classes/Auth.php');
require_once('classes/ConfigManager.php');

$auth = new Auth($pdo);

// Verificar autenticación
if (!$auth->isAuthenticated()) {
    header('Location: login.php');
    exit();
}

// Verificar timeout de sesión
if (!$auth->checkSessionTimeout()) {
    header('Location: login.php?msg=session_expired');
    exit();
}

$configManager = new ConfigManager($pdo, $auth);
$config = $configManager->getConfig();
$historial = $configManager->getHistorial(10);

$error = '';
$success = '';

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $data = $_POST;

    // Procesar subida de archivo si existe
    if (!empty($_FILES['logo']['name'])) {
        $resultado_upload = $configManager->uploadFile($_FILES['logo'], 'logo');
        if ($resultado_upload['success']) {
            $data['url_logo'] = BASE_URL . $resultado_upload['url'];
        } else {
            $error = $resultado_upload['message'];
        }
    }

    if (empty($error)) {
        $resultado = $configManager->updateConfig($data);
        if ($resultado['success']) {
            $success = $resultado['message'];
            $config = $configManager->getConfig();
            $historial = $configManager->getHistorial(10);
        } else {
            $error = $resultado['message'];
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo APP_NAME; ?> - Dashboard</title>
    <link rel="stylesheet" href="css/admin.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <nav class="navbar">
        <div class="navbar-container">
            <div class="navbar-brand">
                <i class="fas fa-radio"></i>
                <span><?php echo APP_NAME; ?></span>
            </div>
            <div class="navbar-menu">
                <span class="user-info">
                    <i class="fas fa-user-circle"></i>
                    <?php echo htmlspecialchars($auth->getCurrentUserEmail()); ?>
                </span>
                <a href="?logout=1" class="logout-btn">
                    <i class="fas fa-sign-out-alt"></i> Cerrar Sesión
                </a>
            </div>
        </div>
    </nav>

    <div class="container">
        <div class="sidebar">
            <ul class="nav-menu">
                <li><a href="#" class="nav-link active"><i class="fas fa-cog"></i> Configuración</a></li>
                <li><a href="#" class="nav-link"><i class="fas fa-history"></i> Historial</a></li>
                <li><a href="#" class="nav-link"><i class="fas fa-users"></i> Usuarios</a></li>
                <li><a href="#" class="nav-link"><i class="fas fa-download"></i> Descargar API</a></li>
            </ul>
        </div>

        <main class="content">
            <!-- Alertas -->
            <?php if ($success): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <?php echo htmlspecialchars($success); ?>
            </div>
            <?php endif; ?>

            <?php if ($error): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-circle"></i>
                <?php echo htmlspecialchars($error); ?>
            </div>
            <?php endif; ?>

            <!-- Formulario de Configuración -->
            <div class="form-card">
                <h2><i class="fas fa-sliders-h"></i> Configuración de la Aplicación</h2>
                
                <form method="POST" enctype="multipart/form-data" class="config-form">
                    <!-- Información General -->
                    <div class="form-section">
                        <h3>Información General</h3>
                        
                        <div class="form-group">
                            <label for="nombre_estacion">Nombre de la Estación</label>
                            <input type="text" id="nombre_estacion" name="nombre_estacion" 
                                   value="<?php echo htmlspecialchars($config['nombre_estacion']); ?>" 
                                   required class="form-control">
                        </div>

                        <div class="form-group">
                            <label for="descripcion">Descripción</label>
                            <textarea id="descripcion" name="descripcion" rows="3" class="form-control"><?php echo htmlspecialchars($config['descripcion']); ?></textarea>
                        </div>

                        <div class="form-group">
                            <label for="url_stream">URL del Stream (Shoutcast/Icecast)</label>
                            <input type="url" id="url_stream" name="url_stream" 
                                   value="<?php echo htmlspecialchars($config['url_stream']); ?>" 
                                   placeholder="http://servidor.com:8000/stream" 
                                   required class="form-control">
                            <small>Ejemplo: http://streaming.server.com:8000/live</small>
                        </div>
                    </div>

                    <!-- Diseño Visual -->
                    <div class="form-section">
                        <h3>Diseño Visual</h3>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="color_primario">Color Primario</label>
                                <div class="color-input-group">
                                    <input type="color" id="color_primario" name="color_primario" 
                                           value="<?php echo substr($config['color_primario'], 3); ?>" 
                                           class="form-control color-picker">
                                    <input type="text" class="form-control color-text" 
                                           value="<?php echo htmlspecialchars($config['color_primario']); ?>">
                                </div>
                            </div>

                            <div class="form-group">
                                <label for="color_secundario">Color Secundario</label>
                                <div class="color-input-group">
                                    <input type="color" id="color_secundario" name="color_secundario" 
                                           value="<?php echo substr($config['color_secundario'], 3); ?>" 
                                           class="form-control color-picker">
                                    <input type="text" class="form-control color-text" 
                                           value="<?php echo htmlspecialchars($config['color_secundario']); ?>">
                                </div>
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="logo">Logo de la Estación</label>
                            <div class="file-upload">
                                <input type="file" id="logo" name="logo" accept="image/*" class="file-input">
                                <label for="logo" class="file-label">
                                    <i class="fas fa-cloud-upload-alt"></i>
                                    <span>Seleccionar imagen</span>
                                </label>
                            </div>
                            <?php if (!empty($config['url_logo'])): ?>
                            <div class="logo-preview">
                                <img src="<?php echo htmlspecialchars($config['url_logo']); ?>" 
                                     alt="Logo actual" class="preview-img">
                                <small>Logo actual</small>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Redes Sociales -->
                    <div class="form-section">
                        <h3>Redes Sociales</h3>
                        
                        <div class="form-group">
                            <label for="url_facebook">
                                <i class="fab fa-facebook"></i> Facebook
                            </label>
                            <input type="url" id="url_facebook" name="url_facebook" 
                                   value="<?php echo htmlspecialchars($config['url_facebook']); ?>" 
                                   placeholder="https://facebook.com/turadio" class="form-control">
                        </div>

                        <div class="form-group">
                            <label for="url_instagram">
                                <i class="fab fa-instagram"></i> Instagram
                            </label>
                            <input type="url" id="url_instagram" name="url_instagram" 
                                   value="<?php echo htmlspecialchars($config['url_instagram']); ?>" 
                                   placeholder="https://instagram.com/turadio" class="form-control">
                        </div>

                        <div class="form-group">
                            <label for="url_tiktok">
                                <i class="fab fa-tiktok"></i> TikTok
                            </label>
                            <input type="url" id="url_tiktok" name="url_tiktok" 
                                   value="<?php echo htmlspecialchars($config['url_tiktok']); ?>" 
                                   placeholder="https://tiktok.com/@turadio" class="form-control">
                        </div>

                        <div class="form-group">
                            <label for="url_whatsapp">
                                <i class="fab fa-whatsapp"></i> WhatsApp
                            </label>
                            <input type="url" id="url_whatsapp" name="url_whatsapp" 
                                   value="<?php echo htmlspecialchars($config['url_whatsapp']); ?>" 
                                   placeholder="https://wa.me/1234567890" class="form-control">
                            <small>Formato: https://wa.me/número (sin espacios)</small>
                        </div>
                    </div>

                    <!-- Botón Guardar -->
                    <div class="form-actions">
                        <button type="submit" class="btn btn-primary btn-lg">
                            <i class="fas fa-save"></i> Guardar Cambios
                        </button>
                        <a href="api/export.php" class="btn btn-secondary">
                            <i class="fas fa-download"></i> Descargar JSON
                        </a>
                    </div>
                </form>
            </div>

            <!-- Historial de Cambios -->
            <div class="form-card">
                <h2><i class="fas fa-history"></i> Historial de Cambios</h2>
                <div class="table-container">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Usuario</th>
                                <th>Tipo de Cambio</th>
                                <th>Descripción</th>
                                <th>Fecha</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($historial as $cambio): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($cambio['usuario_nombre']); ?></td>
                                <td><span class="badge badge-info"><?php echo htmlspecialchars($cambio['tipo_cambio']); ?></span></td>
                                <td><?php echo htmlspecialchars($cambio['descripcion']); ?></td>
                                <td><?php echo date('d/m/Y H:i', strtotime($cambio['fecha_cambio'])); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

        </main>
    </div>

    <script src="js/admin.js"></script>
</body>
</html>
