// admin.js - Funcionalidades del panel administrativo

document.addEventListener('DOMContentLoaded', function() {
    initializeColorPickers();
    initializeFileUpload();
    initializeFormValidation();
});

/**
 * Sincronizar color picker con entrada de texto
 */
function initializeColorPickers() {
    const colorPickers = document.querySelectorAll('.color-picker');
    
    colorPickers.forEach(picker => {
        picker.addEventListener('change', function() {
            const value = '#FF' + this.value.toUpperCase().substring(1);
            const textInput = this.closest('.color-input-group').querySelector('.color-text');
            if (textInput) {
                textInput.value = value;
            }
        });
        
        // Obtener el valor inicial
        const textInput = picker.closest('.color-input-group').querySelector('.color-text');
        if (textInput && textInput.value) {
            // Convertir #FFRRGGBB a #RRGGBB para el input color
            picker.value = '#' + textInput.value.substring(3);
        }
    });
}

/**
 * Mostrar preview de archivo subido
 */
function initializeFileUpload() {
    const fileInput = document.getElementById('logo');
    
    if (fileInput) {
        fileInput.addEventListener('change', function() {
            const file = this.files[0];
            
            if (file) {
                // Validar tipo de archivo
                const allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
                if (!allowedTypes.includes(file.type)) {
                    alert('Por favor selecciona una imagen válida (JPG, PNG, GIF, WebP)');
                    this.value = '';
                    return;
                }
                
                // Validar tamaño (máx 5MB)
                if (file.size > 5 * 1024 * 1024) {
                    alert('El archivo no debe exceder 5MB');
                    this.value = '';
                    return;
                }
                
                // Mostrar preview
                const reader = new FileReader();
                reader.onload = function(e) {
                    let previewContainer = document.querySelector('.logo-preview');
                    
                    if (!previewContainer) {
                        previewContainer = document.createElement('div');
                        previewContainer.className = 'logo-preview';
                        fileInput.closest('.form-group').appendChild(previewContainer);
                    }
                    
                    previewContainer.innerHTML = `
                        <img src="${e.target.result}" alt="Preview" class="preview-img">
                        <small>Preview del nuevo logo</small>
                    `;
                };
                reader.readAsDataURL(file);
            }
        });
    }
}

/**
 * Validación del formulario
 */
function initializeFormValidation() {
    const form = document.querySelector('.config-form');
    
    if (form) {
        form.addEventListener('submit', function(e) {
            const streamUrl = document.getElementById('url_stream').value;
            
            if (!streamUrl.trim()) {
                e.preventDefault();
                alert('Por favor completa la URL del stream');
                return;
            }
            
            // Validar URLs de redes sociales
            const urlFields = [
                'url_facebook',
                'url_instagram',
                'url_tiktok',
                'url_whatsapp'
            ];
            
            let allValid = true;
            urlFields.forEach(fieldId => {
                const field = document.getElementById(fieldId);
                const value = field.value.trim();
                
                if (value && !isValidUrl(value)) {
                    field.classList.add('error');
                    allValid = false;
                } else {
                    field.classList.remove('error');
                }
            });
            
            if (!allValid) {
                e.preventDefault();
                alert('Por favor verifica que todas las URLs sean válidas');
            } else {
                // Mostrar mensaje de guardando
                const submitBtn = form.querySelector('button[type="submit"]');
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Guardando...';
            }
        });
    }
}

/**
 * Validar URL
 */
function isValidUrl(url) {
    try {
        new URL(url);
        return true;
    } catch {
        return false;
    }
}

/**
 * Copiar JSON a portapapeles
 */
function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(() => {
        alert('Configuración copiada al portapapeles');
    }).catch(err => {
        console.error('Error al copiar:', err);
    });
}

/**
 * Descargar JSON
 */
function downloadJSON() {
    const config = {
        stationName: document.getElementById('nombre_estacion').value,
        streamUrl: document.getElementById('url_stream').value,
        primaryColor: document.getElementById('color_primario').value || '#FF6200EE',
        secondaryColor: document.getElementById('color_secundario').value || '#FF03DAC6',
        logoUrl: document.querySelector('.logo-preview img')?.src || '',
        facebookUrl: document.getElementById('url_facebook').value,
        instagramUrl: document.getElementById('url_instagram').value,
        tiktokUrl: document.getElementById('url_tiktok').value,
        whatsappUrl: document.getElementById('url_whatsapp').value,
        description: document.getElementById('descripcion').value
    };
    
    const json = JSON.stringify(config, null, 2);
    const blob = new Blob([json], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'radio_config.json';
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
}

/**
 * Limpiar formulario
 */
function resetForm() {
    const form = document.querySelector('.config-form');
    if (form) {
        form.reset();
    }
}

/**
 * Buscar en tabla
 */
function searchTable(inputId, tableId) {
    const input = document.getElementById(inputId);
    const table = document.getElementById(tableId);
    
    if (!input || !table) return;
    
    input.addEventListener('keyup', function() {
        const searchTerm = this.value.toLowerCase();
        const rows = table.querySelectorAll('tbody tr');
        
        rows.forEach(row => {
            const text = row.textContent.toLowerCase();
            row.style.display = text.includes(searchTerm) ? '' : 'none';
        });
    });
}

/**
 * Exportar tabla a CSV
 */
function exportTableToCSV(tableId, filename) {
    const table = document.getElementById(tableId);
    if (!table) return;
    
    let csv = [];
    const rows = table.querySelectorAll('tr');
    
    rows.forEach(row => {
        const cols = row.querySelectorAll('td, th');
        const rowData = [];
        
        cols.forEach(col => {
            rowData.push('"' + col.textContent.trim().replace(/"/g, '""') + '"');
        });
        
        csv.push(rowData.join(','));
    });
    
    const csvContent = 'data:text/csv;charset=utf-8,' + encodeURIComponent(csv.join('\n'));
    const link = document.createElement('a');
    link.setAttribute('href', csvContent);
    link.setAttribute('download', filename + '.csv');
    link.click();
}

/**
 * Mostrar confirmación antes de eliminar
 */
function confirmDelete(message) {
    return confirm(message || '¿Está seguro de que desea eliminar esto?');
}

/**
 * Crear notificación
 */
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `alert alert-${type}`;
    notification.innerHTML = `
        <i class="fas fa-${type === 'error' ? 'exclamation-circle' : type === 'success' ? 'check-circle' : 'info-circle'}"></i>
        ${message}
    `;
    
    const container = document.querySelector('.content');
    if (container) {
        container.insertBefore(notification, container.firstChild);
        
        // Auto-cerrar después de 5 segundos
        setTimeout(() => {
            notification.remove();
        }, 5000);
    }
}

/**
 * Formatear fecha
 */
function formatDate(dateString) {
    const date = new Date(dateString);
    const formatter = new Intl.DateTimeFormat('es-ES', {
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit'
    });
    return formatter.format(date);
}
