package com.radioapp.streaming.network

import android.content.Context
import android.util.Log
import androidx.security.crypto.EncryptedSharedPreferences
import androidx.security.crypto.MasterKey
import com.google.gson.Gson
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import okhttp3.OkHttpClient
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import retrofit2.http.GET
import java.util.concurrent.TimeUnit

class ConfigManager(private val context: Context) {
    
    private val gson = Gson()
    private val apiClient = ApiClient()
    
    private fun getEncryptedPrefs(): EncryptedSharedPreferences {
        val masterKey = MasterKey.Builder(context)
            .setKeyScheme(MasterKey.KeyScheme.AES256_GCM)
            .build()
        
        return EncryptedSharedPreferences.create(
            context,
            "radio_config",
            masterKey,
            EncryptedSharedPreferences.PrefKeyEncryptionScheme.AES256_SIV,
            EncryptedSharedPreferences.PrefValueEncryptionScheme.AES256_GCM
        ) as EncryptedSharedPreferences
    }

    suspend fun getConfig(): ConfigData = withContext(Dispatchers.IO) {
        return@withContext try {
            val response = apiClient.getRadioConfig()
            
            // Guardar configuración localmente
            saveConfig(response)
            response
        } catch (e: Exception) {
            Log.e("ConfigManager", "Error obteniendo configuración: ${e.message}")
            // Retornar configuración guardada o por defecto
            getLocalConfig() ?: ConfigData()
        }
    }

    fun getLocalConfig(): ConfigData? {
        return try {
            val prefs = getEncryptedPrefs()
            val configJson = prefs.getString("config", null) ?: return null
            gson.fromJson(configJson, ConfigData::class.java)
        } catch (e: Exception) {
            Log.e("ConfigManager", "Error leyendo configuración local: ${e.message}")
            null
        }
    }

    private fun saveConfig(config: ConfigData) {
        try {
            val prefs = getEncryptedPrefs()
            val configJson = gson.toJson(config)
            prefs.edit().putString("config", configJson).apply()
        } catch (e: Exception) {
            Log.e("ConfigManager", "Error guardando configuración: ${e.message}")
        }
    }
}

interface RadioConfigApi {
    @GET("api/config")
    suspend fun getRadioConfig(): ConfigData
}

class ApiClient {
    
    private val okHttpClient = OkHttpClient.Builder()
        .connectTimeout(10, TimeUnit.SECONDS)
        .readTimeout(10, TimeUnit.SECONDS)
        .writeTimeout(10, TimeUnit.SECONDS)
        .build()
    
    private val retrofit = Retrofit.Builder()
        .baseUrl(getApiBaseUrl())
        .addConverterFactory(GsonConverterFactory.create())
        .client(okHttpClient)
        .build()
    
    private val api: RadioConfigApi = retrofit.create(RadioConfigApi::class.java)

    suspend fun getRadioConfig(): ConfigData {
        return api.getRadioConfig()
    }

    companion object {
        fun getApiBaseUrl(): String {
            // Esta URL debe ser configurada en BuildConfig o como variable de entorno
            return "https://tu-dominio.com/"
        }
    }
}

data class ConfigData(
    val stationName: String = "Mi Radio",
    val streamUrl: String = "",
    val primaryColor: String = "#FF6200EE",
    val secondaryColor: String = "#FF03DAC6",
    val logoUrl: String = "",
    val facebookUrl: String = "",
    val instagramUrl: String = "",
    val tiktokUrl: String = "",
    val whatsappUrl: String = "",
    val description: String = "Mi estación de radio"
)
