package com.radioapp.streaming

import android.annotation.SuppressLint
import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.ServiceConnection
import android.net.Uri
import android.os.Bundle
import android.os.IBinder
import android.widget.Button
import android.widget.ImageButton
import android.widget.ImageView
import android.widget.ProgressBar
import android.widget.TextView
import android.widget.Toast
import androidx.appcompat.app.AppCompatActivity
import androidx.core.content.ContextCompat
import com.bumptech.glide.Glide
import com.radioapp.streaming.network.ApiClient
import com.radioapp.streaming.network.ConfigManager
import com.radioapp.streaming.service.MusicService
import com.radioapp.streaming.service.MusicService.PlaybackState
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

class MainActivity : AppCompatActivity() {
    
    private var musicService: MusicService? = null
    private var isServiceBound = false
    
    private lateinit var playPauseBtn: ImageButton
    private lateinit var stopBtn: ImageButton
    private lateinit var stationNameTv: TextView
    private lateinit var songTitleTv: TextView
    private lateinit var streamStatusTv: TextView
    private lateinit var stationLogoIv: ImageView
    private lateinit var loadingProgressBar: ProgressBar
    private lateinit var fbBtn: Button
    private lateinit var igBtn: Button
    private lateinit var tiktokBtn: Button
    private lateinit var whatsappBtn: Button
    
    private val configManager by lazy { ConfigManager(this) }
    
    private val serviceConnection = object : ServiceConnection {
        override fun onServiceConnected(name: ComponentName?, service: IBinder?) {
            val binder = service as MusicService.LocalBinder
            musicService = binder.getService()
            isServiceBound = true
            musicService?.setPlaybackListener(playbackListener)
            updateUIFromService()
        }

        override fun onServiceDisconnected(name: ComponentName?) {
            isServiceBound = false
            musicService = null
        }
    }

    private val playbackListener = object : MusicService.PlaybackListener {
        override fun onPlaybackStateChanged(state: PlaybackState) {
            updatePlayPauseButton(state)
            updateStreamStatus(state)
        }

        override fun onMetadataChanged(title: String, artist: String) {
            songTitleTv.text = artist.ifEmpty { "Transmisión en vivo" }
        }

        override fun onError(error: String) {
            Toast.makeText(this@MainActivity, "Error: $error", Toast.LENGTH_SHORT).show()
            loadingProgressBar.visibility = ProgressBar.GONE
        }

        override fun onBuffering() {
            loadingProgressBar.visibility = ProgressBar.VISIBLE
        }

        override fun onBufferingComplete() {
            loadingProgressBar.visibility = ProgressBar.GONE
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_main)
        
        initializeViews()
        loadConfiguration()
        bindMusicService()
    }

    private fun initializeViews() {
        playPauseBtn = findViewById(R.id.play_pause_btn)
        stopBtn = findViewById(R.id.stop_btn)
        stationNameTv = findViewById(R.id.station_name)
        songTitleTv = findViewById(R.id.song_title)
        streamStatusTv = findViewById(R.id.stream_status)
        stationLogoIv = findViewById(R.id.station_logo)
        loadingProgressBar = findViewById(R.id.loading_progress)
        fbBtn = findViewById(R.id.fb_btn)
        igBtn = findViewById(R.id.ig_btn)
        tiktokBtn = findViewById(R.id.tiktok_btn)
        whatsappBtn = findViewById(R.id.whatsapp_btn)

        playPauseBtn.setOnClickListener { togglePlayPause() }
        stopBtn.setOnClickListener { stopPlayback() }
        fbBtn.setOnClickListener { openSocialMedia("facebook") }
        igBtn.setOnClickListener { openSocialMedia("instagram") }
        tiktokBtn.setOnClickListener { openSocialMedia("tiktok") }
        whatsappBtn.setOnClickListener { openSocialMedia("whatsapp") }
    }

    private fun loadConfiguration() {
        CoroutineScope(Dispatchers.Main).launch {
            try {
                val config = configManager.getConfig()
                applyConfiguration(config)
            } catch (e: Exception) {
                Toast.makeText(this@MainActivity, "Error cargando configuración", Toast.LENGTH_SHORT).show()
            }
        }
    }

    @SuppressLint("SetTextI18n")
    private fun applyConfiguration(config: ConfigData) {
        stationNameTv.text = config.stationName
        stationNameTv.setTextColor(android.graphics.Color.parseColor(config.primaryColor))
        
        songTitleTv.setTextColor(android.graphics.Color.parseColor(config.secondaryColor))
        
        window.statusBarColor = android.graphics.Color.parseColor(config.primaryColor)
        
        if (config.logoUrl.isNotEmpty()) {
            Glide.with(this)
                .load(config.logoUrl)
                .placeholder(R.drawable.ic_radio_placeholder)
                .error(R.drawable.ic_radio_placeholder)
                .into(stationLogoIv)
        }

        musicService?.setStreamUrl(config.streamUrl)
        
        updateSocialButtons(config)
    }

    private fun updateSocialButtons(config: ConfigData) {
        fbBtn.setOnClickListener { openUrl(config.facebookUrl) }
        igBtn.setOnClickListener { openUrl(config.instagramUrl) }
        tiktokBtn.setOnClickListener { openUrl(config.tiktokUrl) }
        whatsappBtn.setOnClickListener { openUrl(config.whatsappUrl) }
    }

    private fun openSocialMedia(platform: String) {
        val config = configManager.getLocalConfig() ?: return
        val url = when (platform) {
            "facebook" -> config.facebookUrl
            "instagram" -> config.instagramUrl
            "tiktok" -> config.tiktokUrl
            "whatsapp" -> config.whatsappUrl
            else -> return
        }
        if (url.isNotEmpty()) openUrl(url)
    }

    private fun openUrl(url: String) {
        try {
            val intent = Intent(Intent.ACTION_VIEW, Uri.parse(url))
            startActivity(intent)
        } catch (e: Exception) {
            Toast.makeText(this, "No se puede abrir el enlace", Toast.LENGTH_SHORT).show()
        }
    }

    private fun togglePlayPause() {
        if (musicService?.isPlaying() == true) {
            musicService?.pause()
        } else {
            musicService?.play()
        }
    }

    private fun stopPlayback() {
        musicService?.stop()
    }

    private fun updatePlayPauseButton(state: PlaybackState) {
        runOnUiThread {
            playPauseBtn.setImageResource(
                if (state == PlaybackState.PLAYING) R.drawable.ic_pause else R.drawable.ic_play
            )
        }
    }

    private fun updateStreamStatus(state: PlaybackState) {
        val status = when (state) {
            PlaybackState.PLAYING -> "En vivo"
            PlaybackState.PAUSED -> "Pausado"
            PlaybackState.STOPPED -> "Detenido"
            PlaybackState.BUFFERING -> "Cargando..."
            PlaybackState.ERROR -> "Error"
        }
        runOnUiThread { streamStatusTv.text = status }
    }

    private fun updateUIFromService() {
        val state = musicService?.getCurrentState() ?: PlaybackState.STOPPED
        updatePlayPauseButton(state)
        updateStreamStatus(state)
    }

    private fun bindMusicService() {
        Intent(this, MusicService::class.java).also { intent ->
            bindService(intent, serviceConnection, Context.BIND_AUTO_CREATE)
            startService(intent)
        }
    }

    override fun onStart() {
        super.onStart()
        if (!isServiceBound) {
            bindMusicService()
        }
    }

    override fun onStop() {
        super.onStop()
        if (isServiceBound) {
            musicService?.setPlaybackListener(null)
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        if (isServiceBound) {
            unbindService(serviceConnection)
        }
    }
}

data class ConfigData(
    val stationName: String = "Mi Radio",
    val streamUrl: String = "",
    val primaryColor: String = "#FF6200EE",
    val secondaryColor: String = "#FF03DAC6",
    val logoUrl: String = "",
    val facebookUrl: String = "",
    val instagramUrl: String = "",
    val tiktokUrl: String = "",
    val whatsappUrl: String = ""
)
