package com.radioapp.streaming.service

import android.app.Service
import android.content.Intent
import android.media.MediaPlayer
import android.os.Binder
import android.os.IBinder
import android.util.Log
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.io.IOException

class MusicService : Service() {
    
    private var mediaPlayer: MediaPlayer? = null
    private var currentStreamUrl: String = ""
    private var currentState: PlaybackState = PlaybackState.STOPPED
    private var playbackListener: PlaybackListener? = null
    private val binder = LocalBinder()

    inner class LocalBinder : Binder() {
        fun getService(): MusicService = this@MusicService
    }

    enum class PlaybackState {
        PLAYING, PAUSED, STOPPED, BUFFERING, ERROR
    }

    interface PlaybackListener {
        fun onPlaybackStateChanged(state: PlaybackState)
        fun onMetadataChanged(title: String, artist: String)
        fun onError(error: String)
        fun onBuffering()
        fun onBufferingComplete()
    }

    override fun onBind(intent: Intent): IBinder = binder

    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
        return START_STICKY
    }

    fun setStreamUrl(url: String) {
        if (currentStreamUrl != url) {
            currentStreamUrl = url
            if (currentState == PlaybackState.PLAYING) {
                stop()
                play()
            }
        }
    }

    fun play() {
        if (currentStreamUrl.isEmpty()) {
            notifyError("URL del stream no configurada")
            return
        }

        if (currentState == PlaybackState.PLAYING) return
        if (currentState == PlaybackState.PAUSED) {
            mediaPlayer?.start()
            updateState(PlaybackState.PLAYING)
            return
        }

        initializeMediaPlayer()
        prepareAndPlay()
    }

    fun pause() {
        mediaPlayer?.pause()
        updateState(PlaybackState.PAUSED)
    }

    fun stop() {
        mediaPlayer?.stop()
        mediaPlayer?.release()
        mediaPlayer = null
        updateState(PlaybackState.STOPPED)
    }

    fun isPlaying(): Boolean = currentState == PlaybackState.PLAYING

    fun getCurrentState(): PlaybackState = currentState

    fun setPlaybackListener(listener: PlaybackListener?) {
        playbackListener = listener
    }

    private fun initializeMediaPlayer() {
        if (mediaPlayer != null) {
            mediaPlayer?.release()
        }
        mediaPlayer = MediaPlayer().apply {
            setOnPreparedListener { onMediaPlayerPrepared() }
            setOnErrorListener { _, what, extra ->
                val errorMsg = "Error: $what - $extra"
                Log.e("MusicService", errorMsg)
                notifyError(errorMsg)
                true
            }
            setOnCompletionListener { stop() }
            setOnBufferingUpdateListener { _, _ ->
                playbackListener?.onBufferingComplete()
            }
        }
    }

    private fun prepareAndPlay() {
        CoroutineScope(Dispatchers.IO).launch {
            try {
                updateState(PlaybackState.BUFFERING)
                playbackListener?.onBuffering()
                
                mediaPlayer?.apply {
                    setDataSource(currentStreamUrl)
                    prepareAsync()
                }
            } catch (e: IOException) {
                Log.e("MusicService", "Error preparando stream", e)
                notifyError("Error: ${e.message}")
            }
        }
    }

    private fun onMediaPlayerPrepared() {
        try {
            mediaPlayer?.start()
            updateState(PlaybackState.PLAYING)
            playbackListener?.onBufferingComplete()
            extractAndNotifyMetadata()
        } catch (e: Exception) {
            Log.e("MusicService", "Error al iniciar reproducción", e)
            notifyError("Error al iniciar reproducción")
        }
    }

    private fun extractAndNotifyMetadata() {
        try {
            val metadata = mediaPlayer?.let { player ->
                // Obtener información del stream (si está disponible)
                "Transmisión en vivo"
            } ?: "Transmisión en vivo"
            
            playbackListener?.onMetadataChanged(metadata, "")
        } catch (e: Exception) {
            Log.e("MusicService", "Error extrayendo metadata", e)
        }
    }

    private fun updateState(newState: PlaybackState) {
        currentState = newState
        playbackListener?.onPlaybackStateChanged(newState)
    }

    private fun notifyError(error: String) {
        updateState(PlaybackState.ERROR)
        playbackListener?.onError(error)
    }

    override fun onDestroy() {
        super.onDestroy()
        stop()
    }
}
