-- ========================================
-- Base de Datos: radio_admin
-- Descripción: Base de datos para gestionar la configuración de la radio
-- ========================================

CREATE DATABASE IF NOT EXISTS radio_admin CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE radio_admin;

-- ========================================
-- Tabla: usuarios
-- Descripción: Usuarios del panel de administración
-- ========================================
CREATE TABLE IF NOT EXISTS usuarios (
    id INT PRIMARY KEY AUTO_INCREMENT,
    nombre VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    rol ENUM('admin', 'editor') DEFAULT 'editor',
    activo BOOLEAN DEFAULT TRUE,
    fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ========================================
-- Tabla: configuracion
-- Descripción: Configuración principal de la aplicación
-- ========================================
CREATE TABLE IF NOT EXISTS configuracion (
    id INT PRIMARY KEY AUTO_INCREMENT,
    nombre_estacion VARCHAR(100) NOT NULL,
    descripcion TEXT,
    url_stream VARCHAR(500) NOT NULL,
    color_primario VARCHAR(7) DEFAULT '#FF6200EE',
    color_secundario VARCHAR(7) DEFAULT '#FF03DAC6',
    url_logo VARCHAR(500),
    url_facebook VARCHAR(500),
    url_instagram VARCHAR(500),
    url_tiktok VARCHAR(500),
    url_whatsapp VARCHAR(500),
    version_app VARCHAR(20) DEFAULT '1.0.0',
    activo BOOLEAN DEFAULT TRUE,
    fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ========================================
-- Tabla: historial_cambios
-- Descripción: Registro de cambios realizados en la configuración
-- ========================================
CREATE TABLE IF NOT EXISTS historial_cambios (
    id INT PRIMARY KEY AUTO_INCREMENT,
    usuario_id INT NOT NULL,
    tipo_cambio VARCHAR(50) NOT NULL,
    descripcion TEXT,
    valor_anterior JSON,
    valor_nuevo JSON,
    fecha_cambio TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ========================================
-- Tabla: sesiones_api
-- Descripción: Tokens de API para acceso desde la app
-- ========================================
CREATE TABLE IF NOT EXISTS sesiones_api (
    id INT PRIMARY KEY AUTO_INCREMENT,
    usuario_id INT NOT NULL,
    token VARCHAR(255) UNIQUE NOT NULL,
    ip_origen VARCHAR(45),
    user_agent TEXT,
    fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    fecha_expiracion TIMESTAMP NOT NULL,
    activo BOOLEAN DEFAULT TRUE,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ========================================
-- Tabla: logs_acceso
-- Descripción: Registro de accesos al panel
-- ========================================
CREATE TABLE IF NOT EXISTS logs_acceso (
    id INT PRIMARY KEY AUTO_INCREMENT,
    usuario_id INT,
    accion VARCHAR(100) NOT NULL,
    ip_origen VARCHAR(45),
    user_agent TEXT,
    estado VARCHAR(20),
    fecha TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ========================================
-- Inserciones iniciales
-- ========================================

-- Usuario admin por defecto (contraseña: admin123)
INSERT INTO usuarios (nombre, email, password, rol, activo) VALUES
('Administrador', 'admin@radioapp.com', '$2y$10$abcdefghijklmnopqrstuvwxyz0123456789', 'admin', TRUE);

-- Configuración inicial
INSERT INTO configuracion (
    nombre_estacion, 
    descripcion, 
    url_stream, 
    color_primario, 
    color_secundario, 
    url_logo,
    url_facebook,
    url_instagram,
    url_tiktok,
    url_whatsapp
) VALUES (
    'Mi Estación de Radio',
    'Una estación de radio moderna y dinámica',
    'http://tu-servidor.com:8000/stream',
    '#FF6200EE',
    '#FF03DAC6',
    'https://tu-dominio.com/logo.png',
    'https://facebook.com/turadio',
    'https://instagram.com/turadio',
    'https://tiktok.com/@turadio',
    'https://wa.me/1234567890'
);

-- Índices
CREATE INDEX idx_usuario_email ON usuarios(email);
CREATE INDEX idx_configuracion_activo ON configuracion(activo);
CREATE INDEX idx_historial_usuario ON historial_cambios(usuario_id);
CREATE INDEX idx_historial_fecha ON historial_cambios(fecha_cambio);
CREATE INDEX idx_sesion_token ON sesiones_api(token);
CREATE INDEX idx_logs_usuario ON logs_acceso(usuario_id);
CREATE INDEX idx_logs_fecha ON logs_acceso(fecha);
