# CONFIGURACIÓN DE SERVIDOR

## ====== APACHE (.htaccess) ======

Guardar como: admin-panel/.htaccess

```apache
# Habilitar módulo rewrite
<IfModule mod_rewrite.c>
    RewriteEngine On
    RewriteBase /admin-panel/
    
    # Redirigir HTTP a HTTPS
    RewriteCond %{HTTPS} off
    RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
    
    # Permitir archivos y carpetas reales
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteCond %{REQUEST_FILENAME} !-d
    
    # Redirigir solicitudes a index.php
    RewriteRule ^(.*)$ index.php?path=$1 [QSA,L]
</IfModule>

# Proteger archivos sensibles
<Files "config.php">
    Deny from all
</Files>

<Files "*.sql">
    Deny from all
</Files>

# Headers de seguridad
<IfModule mod_headers.c>
    Header set X-Content-Type-Options "nosniff"
    Header set X-Frame-Options "SAMEORIGIN"
    Header set X-XSS-Protection "1; mode=block"
    Header set Referrer-Policy "strict-origin-when-cross-origin"
    Header set Content-Security-Policy "default-src 'self'; script-src 'self' 'unsafe-inline' cdnjs.cloudflare.com; style-src 'self' 'unsafe-inline' cdnjs.cloudflare.com; font-src 'self' cdnjs.cloudflare.com"
</IfModule>

# Optimización
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/plain text/xml text/css text/javascript application/javascript
</IfModule>

<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType text/css "access plus 1 year"
    ExpiresByType application/javascript "access plus 1 year"
    ExpiresByType image/jpeg "access plus 1 month"
    ExpiresByType image/png "access plus 1 month"
    ExpiresByType image/gif "access plus 1 month"
</IfModule>

# CORS
<IfModule mod_headers.c>
    Header set Access-Control-Allow-Origin "*"
    Header set Access-Control-Allow-Methods "GET, POST, PUT, DELETE, OPTIONS"
    Header set Access-Control-Allow-Headers "Content-Type, Authorization"
</IfModule>

# Límite de tamaño de archivo
php_value upload_max_filesize 50M
php_value post_max_size 50M
```

---

## ====== NGINX ======

Guardar como: /etc/nginx/sites-available/radio-app

```nginx
# HTTP a HTTPS
server {
    listen 80;
    server_name tu-dominio.com www.tu-dominio.com;
    
    return 301 https://$server_name$request_uri;
}

# HTTPS
server {
    listen 443 ssl http2;
    server_name tu-dominio.com www.tu-dominio.com;

    # Certificados SSL (Let's Encrypt)
    ssl_certificate /etc/letsencrypt/live/tu-dominio.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/tu-dominio.com/privkey.pem;
    
    # Configuración SSL
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_ciphers 'ECDHE-RSA-AES128-GCM-SHA256:ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-AES256-GCM-SHA384';
    ssl_prefer_server_ciphers on;
    ssl_session_cache shared:SSL:10m;
    ssl_session_timeout 10m;

    # Raíz del proyecto
    root /var/www/html;
    index index.php;

    # Logs
    access_log /var/log/nginx/radio-app-access.log;
    error_log /var/log/nginx/radio-app-error.log;

    # Límites
    client_max_body_size 50M;

    # Headers de seguridad
    add_header X-Content-Type-Options "nosniff";
    add_header X-Frame-Options "SAMEORIGIN";
    add_header X-XSS-Protection "1; mode=block";
    add_header Referrer-Policy "strict-origin-when-cross-origin";

    # Gzip
    gzip on;
    gzip_vary on;
    gzip_types text/plain text/css text/javascript application/javascript;

    # Proxy al panel administrativo
    location /admin-panel/ {
        try_files $uri $uri/ /admin-panel/index.php?path=$uri;
    }

    # API REST
    location /api/ {
        try_files $uri $uri/ /api/index.php?path=$uri;
    }

    # Archivos estáticos
    location ~ \.(js|css|png|jpg|jpeg|gif|ico|svg|woff|woff2|ttf)$ {
        expires 1y;
        add_header Cache-Control "public, immutable";
    }

    # Procesar PHP
    location ~ \.php$ {
        include snippets/fastcgi-php.conf;
        fastcgi_pass unix:/var/run/php/php8.2-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
    }

    # Negar acceso a archivos sensibles
    location ~ /\. {
        deny all;
    }

    location ~ ~$ {
        deny all;
    }

    location ~ /config\.php$ {
        deny all;
    }

    location ~ /\.sql$ {
        deny all;
    }

    # Redirecionar /index.php
    if ($request_uri ~ "^/index\.php") {
        return 301 /;
    }

    # CORS para API
    if ($request_method = 'OPTIONS') {
        add_header 'Access-Control-Allow-Origin' '*';
        add_header 'Access-Control-Allow-Methods' 'GET, POST, PUT, DELETE, OPTIONS';
        add_header 'Access-Control-Allow-Headers' 'Content-Type, Authorization';
        add_header 'Access-Control-Max-Age' 1728000;
        return 204;
    }
}
```

### Habilitar sitio en Nginx:

```bash
# Crear enlace simbólico
sudo ln -s /etc/nginx/sites-available/radio-app /etc/nginx/sites-enabled/

# Verificar configuración
sudo nginx -t

# Reiniciar Nginx
sudo systemctl restart nginx
```

---

## ====== CERTIFICADO SSL (Let's Encrypt) ======

```bash
# Instalar Certbot
sudo apt-get install certbot python3-certbot-nginx

# Obtener certificado
sudo certbot certonly --nginx -d tu-dominio.com -d www.tu-dominio.com

# Auto-renovación (verificar)
sudo systemctl enable certbot.timer
sudo systemctl start certbot.timer

# Ver certificados instalados
sudo certbot certificates
```

---

## ====== CONFIGURACIÓN PHP (php.ini) ======

```ini
; Máximo tamaño de subida
upload_max_filesize = 50M
post_max_size = 50M

; Máximo tiempo de ejecución
max_execution_time = 300
max_input_time = 300

; Memory limit
memory_limit = 256M

; Seguridad
expose_php = Off
disable_functions = exec, passthru, shell_exec, system, proc_open, popen, curl_exec, curl_multi_exec, parse_ini_file, show_source
allow_url_fopen = Off
allow_url_include = Off

; Session
session.cookie_secure = On
session.cookie_httponly = On
session.cookie_samesite = "Strict"

; Timezone
date.timezone = America/Lima
```

---

## ====== CREAR USUARIO MYSQL ======

```bash
# Conectar a MySQL
mysql -u root -p

# Ejecutar en MySQL:
CREATE USER 'radio_user'@'localhost' IDENTIFIED BY 'password_seguro_aqui';
GRANT ALL PRIVILEGES ON radio_admin.* TO 'radio_user'@'localhost';
FLUSH PRIVILEGES;
EXIT;
```

---

## ====== BACKUP DE BASE DE DATOS ======

```bash
# Crear backup manual
mysqldump -u radio_user -p radio_admin > backup_$(date +%Y%m%d_%H%M%S).sql

# Script de backup automático (cron)
# Crear archivo: /usr/local/bin/backup-radio.sh

#!/bin/bash
BACKUP_DIR="/backups/radio"
DATE=$(date +%Y%m%d_%H%M%S)
DB_NAME="radio_admin"
DB_USER="radio_user"
DB_PASS="password_seguro"

mkdir -p $BACKUP_DIR

mysqldump -u $DB_USER -p$DB_PASS $DB_NAME > $BACKUP_DIR/backup_$DATE.sql

# Comprimir
gzip $BACKUP_DIR/backup_$DATE.sql

# Eliminar backups anteriores a 30 días
find $BACKUP_DIR -type f -name "*.sql.gz" -mtime +30 -delete

# Hacer ejecutable
chmod +x /usr/local/bin/backup-radio.sh

# Agregar a crontab (ejecutar diariamente a las 2 AM)
# crontab -e
# 0 2 * * * /usr/local/bin/backup-radio.sh
```

---

## ====== MONITOREO ======

```bash
# Ver logs en tiempo real (Apache)
tail -f /var/log/apache2/error.log

# Ver logs en tiempo real (Nginx)
tail -f /var/log/nginx/radio-app-error.log

# Ver logs de PHP
tail -f /var/log/php-fpm.log

# Ver logs de MySQL
tail -f /var/log/mysql/error.log

# Monitoreo de recursos
htop
free -h
df -h
```

---

## ====== FIREWALL (UFW) ======

```bash
# Habilitar UFW
sudo ufw enable

# Permitir SSH
sudo ufw allow 22/tcp

# Permitir HTTP
sudo ufw allow 80/tcp

# Permitir HTTPS
sudo ufw allow 443/tcp

# Permitir MySQL (solo localhost)
sudo ufw allow from 127.0.0.1 to 127.0.0.1 port 3306

# Ver estado
sudo ufw status verbose
```

---

## ====== VERIFICACIÓN FINAL ======

```bash
# Verificar que el sitio está activo
curl -I https://tu-dominio.com/admin-panel/

# Verificar SSL
openssl s_client -connect tu-dominio.com:443 -servername tu-dominio.com

# Probar API
curl https://tu-dominio.com/api/config

# Verificar logs
grep ERROR /var/log/php-fpm.log
grep ERROR /var/log/nginx/radio-app-error.log
```

---

**✅ Si todo funciona correctamente, tu servidor está listo para producción.**
