#!/bin/bash

# ============================================
# RadioApp - Script de Utilidades
# ============================================

set -e

# Colores
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Funciones de utilidad
print_header() {
    echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${BLUE}$1${NC}"
    echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
}

print_success() {
    echo -e "${GREEN}✓ $1${NC}"
}

print_error() {
    echo -e "${RED}✗ $1${NC}"
}

print_warning() {
    echo -e "${YELLOW}⚠ $1${NC}"
}

print_info() {
    echo -e "${BLUE}ℹ $1${NC}"
}

# ============================================
# MENÚ PRINCIPAL
# ============================================

show_menu() {
    print_header "RadioApp - Utilidades"
    echo ""
    echo "1. Compilar APK Debug"
    echo "2. Compilar APK Release"
    echo "3. Instalar APK en dispositivo"
    echo "4. Ver logs de la app"
    echo "5. Crear base de datos"
    echo "6. Hacer backup de BD"
    echo "7. Validar servidor web"
    echo "8. Generar claves de API"
    echo "9. Crear usuario admin"
    echo "10. Limpiar caché"
    echo "11. Ver documentación"
    echo "12. Salir"
    echo ""
}

# ============================================
# FUNCIONES ANDROID
# ============================================

build_apk_debug() {
    print_header "Compilando APK Debug"
    cd android
    
    if [ ! -f "build.gradle.kts" ]; then
        print_error "No se encontró build.gradle.kts"
        return 1
    fi
    
    print_info "Compilando proyecto Android..."
    ./gradlew clean assembleDebug 2>&1 | tail -20
    
    APK_PATH="app/build/outputs/apk/debug/app-debug.apk"
    if [ -f "$APK_PATH" ]; then
        print_success "APK Debug compilado: $APK_PATH"
        print_info "Tamaño: $(du -h $APK_PATH | cut -f1)"
    else
        print_error "Error compilando APK"
        return 1
    fi
    
    cd ..
}

build_apk_release() {
    print_header "Compilando APK Release"
    cd android
    
    if [ ! -f "build.gradle.kts" ]; then
        print_error "No se encontró build.gradle.kts"
        return 1
    fi
    
    print_warning "Asegúrate de tener configurado el keystore"
    print_info "Compilando proyecto Android..."
    ./gradlew clean assembleRelease 2>&1 | tail -20
    
    APK_PATH="app/build/outputs/apk/release/app-release.apk"
    if [ -f "$APK_PATH" ]; then
        print_success "APK Release compilado: $APK_PATH"
        print_info "Tamaño: $(du -h $APK_PATH | cut -f1)"
    else
        print_error "Error compilando APK"
        return 1
    fi
    
    cd ..
}

install_apk() {
    print_header "Instalar APK en dispositivo"
    
    # Buscar APK más reciente
    APK_DEBUG="android/app/build/outputs/apk/debug/app-debug.apk"
    
    if [ ! -f "$APK_DEBUG" ]; then
        print_error "No se encontró APK Debug. Compila primero."
        return 1
    fi
    
    print_info "Dispositivos disponibles:"
    adb devices
    echo ""
    
    if ! adb devices | grep -q "device$"; then
        print_error "No hay dispositivos conectados"
        return 1
    fi
    
    print_info "Instalando APK..."
    adb install "$APK_DEBUG"
    print_success "APK instalado"
}

view_logs() {
    print_header "Logs de la aplicación"
    print_info "Mostrando últimas líneas. Presiona Ctrl+C para salir."
    echo ""
    adb logcat | grep -E "(RadioApp|MusicService|ConfigManager|radio)" || echo "No hay logs"
}

# ============================================
# FUNCIONES BASE DE DATOS
# ============================================

create_database() {
    print_header "Crear base de datos"
    
    if [ ! -f "database/schema.sql" ]; then
        print_error "No se encontró database/schema.sql"
        return 1
    fi
    
    read -p "Usuario MySQL: " db_user
    read -s -p "Contraseña MySQL: " db_pass
    echo ""
    
    print_info "Importando schema..."
    mysql -u "$db_user" -p"$db_pass" < database/schema.sql 2>/dev/null
    
    if [ $? -eq 0 ]; then
        print_success "Base de datos creada"
        print_info "Tablas creadas:"
        mysql -u "$db_user" -p"$db_pass" -e "USE radio_admin; SHOW TABLES;" 2>/dev/null
    else
        print_error "Error al crear base de datos"
        return 1
    fi
}

backup_database() {
    print_header "Hacer backup de la base de datos"
    
    read -p "Usuario MySQL: " db_user
    read -s -p "Contraseña MySQL: " db_pass
    echo ""
    
    BACKUP_FILE="backup_radio_$(date +%Y%m%d_%H%M%S).sql"
    
    print_info "Creando backup en $BACKUP_FILE..."
    mysqldump -u "$db_user" -p"$db_pass" radio_admin > "$BACKUP_FILE" 2>/dev/null
    
    if [ $? -eq 0 ]; then
        gzip "$BACKUP_FILE"
        print_success "Backup creado: ${BACKUP_FILE}.gz"
        print_info "Tamaño: $(du -h ${BACKUP_FILE}.gz | cut -f1)"
    else
        print_error "Error al crear backup"
        return 1
    fi
}

# ============================================
# FUNCIONES SERVIDOR
# ============================================

validate_server() {
    print_header "Validar servidor web"
    
    read -p "Dominio (ej: tu-dominio.com): " domain
    
    print_info "Validando HTTPS..."
    if curl -s -I https://$domain/admin-panel/ | grep -q "200\|301\|302"; then
        print_success "HTTPS funciona"
    else
        print_warning "Error en HTTPS"
    fi
    
    print_info "Validando API..."
    RESPONSE=$(curl -s https://$domain/api/config | grep -c "stationName" || echo "0")
    if [ "$RESPONSE" -gt 0 ]; then
        print_success "API funciona"
    else
        print_warning "Error en API"
    fi
    
    print_info "Validando SSL..."
    echo | openssl s_client -servername $domain -connect $domain:443 2>/dev/null | grep -q "Verify return code: 0"
    if [ $? -eq 0 ]; then
        print_success "SSL válido"
    else
        print_warning "Error SSL"
    fi
}

generate_api_keys() {
    print_header "Generar claves de API"
    
    print_info "Generando token JWT..."
    JWT_SECRET=$(openssl rand -hex 32)
    print_success "JWT_SECRET: $JWT_SECRET"
    
    print_info "Copiar a config.php:"
    echo "define('JWT_SECRET', '$JWT_SECRET');"
}

create_admin_user() {
    print_header "Crear usuario administrador"
    
    read -p "Email: " email
    read -p "Nombre: " nombre
    read -s -p "Contraseña: " password
    echo ""
    
    read -p "Usuario MySQL: " db_user
    read -s -p "Contraseña MySQL: " db_pass
    echo ""
    
    HASH=$(php -r "echo password_hash('$password', PASSWORD_BCRYPT);")
    
    SQL="INSERT INTO usuarios (nombre, email, password, rol, activo) VALUES ('$nombre', '$email', '$HASH', 'admin', 1);"
    
    mysql -u "$db_user" -p"$db_pass" radio_admin -e "$SQL" 2>/dev/null
    
    if [ $? -eq 0 ]; then
        print_success "Usuario creado: $email"
    else
        print_error "Error al crear usuario"
        return 1
    fi
}

# ============================================
# FUNCIONES LIMPIEZA
# ============================================

clean_cache() {
    print_header "Limpiar caché"
    
    print_info "Limpiando caché de Gradle..."
    cd android && ./gradlew clean && cd ..
    print_success "Caché Gradle limpiado"
    
    print_info "Limpiando caché de compilación..."
    rm -rf android/.gradle android/app/build
    print_success "Caché de compilación limpiado"
    
    print_success "Limpieza completada"
}

# ============================================
# FUNCIONES DOCUMENTACIÓN
# ============================================

show_docs() {
    print_header "Documentación disponible"
    
    echo "1. Guía Completa (GUIA_COMPLETA.md)"
    echo "2. Preguntas Frecuentes (FAQ.md)"
    echo "3. Configuración del Servidor (CONFIGURACION_SERVIDOR.md)"
    echo "4. README.md"
    echo ""
    
    read -p "Selecciona una opción (1-4): " choice
    
    case $choice in
        1) less docs/GUIA_COMPLETA.md ;;
        2) less docs/FAQ.md ;;
        3) less docs/CONFIGURACION_SERVIDOR.md ;;
        4) less README.md ;;
        *) print_error "Opción no válida" ;;
    esac
}

# ============================================
# PROGRAMA PRINCIPAL
# ============================================

main() {
    while true; do
        show_menu
        read -p "Selecciona una opción (1-12): " choice
        
        echo ""
        
        case $choice in
            1) build_apk_debug ;;
            2) build_apk_release ;;
            3) install_apk ;;
            4) view_logs ;;
            5) create_database ;;
            6) backup_database ;;
            7) validate_server ;;
            8) generate_api_keys ;;
            9) create_admin_user ;;
            10) clean_cache ;;
            11) show_docs ;;
            12) 
                print_success "¡Hasta luego!"
                exit 0
                ;;
            *)
                print_error "Opción no válida"
                ;;
        esac
        
        echo ""
        read -p "Presiona Enter para continuar..."
        clear
    done
}

# Verificar requisitos
check_requirements() {
    print_header "Verificando requisitos"
    
    local missing=0
    
    # Verificar Android SDK
    if ! command -v adb &> /dev/null; then
        print_warning "adb no encontrado (Android SDK Tools)"
        missing=$((missing + 1))
    else
        print_success "adb disponible"
    fi
    
    # Verificar MySQL
    if ! command -v mysql &> /dev/null; then
        print_warning "mysql no encontrado"
        missing=$((missing + 1))
    else
        print_success "mysql disponible"
    fi
    
    # Verificar PHP
    if ! command -v php &> /dev/null; then
        print_warning "php no encontrado"
        missing=$((missing + 1))
    else
        print_success "php disponible"
    fi
    
    # Verificar OpenSSL
    if ! command -v openssl &> /dev/null; then
        print_warning "openssl no encontrado"
        missing=$((missing + 1))
    else
        print_success "openssl disponible"
    fi
    
    echo ""
    
    if [ $missing -eq 0 ]; then
        print_success "Todos los requisitos están instalados"
    else
        print_warning "$missing requisito(s) faltante(s)"
        print_info "Algunos comandos pueden no funcionar"
    fi
    
    echo ""
    read -p "Presiona Enter para continuar..."
    clear
}

# Ejecutar
check_requirements
main
